unit PolyLine;

interface

uses
{$ifdef VER150}
  DesignIntf, DesignEditors, DesignWindows, DsnConst,
{$else}
  DsgnIntf,
{$endif}
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, stdctrls;


type
  TPolyLine = class(TGraphicControl)
  private
    FData: TStringList;

    FLineColor: TColor;
    FPenStyle: TPenStyle;

    procedure SetupData(Value: TStringList);
    procedure LoadSetupData;
    procedure SaveSetupData;
    procedure SetLineColor(Value: TColor);
    procedure SetPenStyle(Value: TPenStyle);
  protected
    procedure Loaded; override;
    procedure Paint; override;
  public
    FlashLine: integer;
    Line: array of PRect;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure FreeLines;

  published
    property Data: TStringList read FData write SetupData;
    property LineColor: TColor read FLineColor write SetLineColor;
    property PenStyle: TPenStyle read FPenStyle write SetPenStyle;
    property OnMouseDown;
  end;


  TPolyLineEditor = class(TComponentEditor)
  private
  public
    procedure Edit; override;
  end;


procedure Register;

implementation
uses PolyLineSetup;

procedure Register;
begin
  RegisterComponents('Scada', [TPolyLine]);
  RegisterComponentEditor(TPolyLine, TPolyLineEditor);
end;



{ TPolyLine }

constructor TPolyLine.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 65;
  Height := 17;
  Canvas.Brush.Style := bsClear;
  FlashLine := -1;
  FData := TStringList.Create;
end;

destructor TPolyLine.Destroy;
begin
  FreeLines;
  inherited;
end;


procedure TPolyLine.SetupData(Value: TStringList);
begin
  FData.Assign(Value);
  LoadSetupData;
end;


procedure TPolyLine.SetLineColor(Value: TColor);
begin
  FLineColor := Value;
  Invalidate;
end;


procedure TPolyLine.SetPenStyle(Value: TPenStyle);
begin
  FPenStyle := value;
  Invalidate;
end;

procedure TPolyLine.Paint;
var
  i: integer;
begin

  Canvas.Pen.Color := FLineColor;
  Canvas.Pen.Style := FPenStyle;

  for i:=0 to length(Line)-1 do begin
    if FlashLine=i then Canvas.Pen.Color := $FFFFFF-FLineColor;
    Canvas.MoveTo(Line[i].Left, Line[i].Top);
    Canvas.LineTo(Line[i].Right, Line[i].Bottom);
    if FlashLine=i then Canvas.Pen.Color := FLineColor;
  end;
end;


procedure TPolyLine.LoadSetupData;
var
  i,k,v,j,n: integer;
  s: string;
begin
  FreeLines;
  
  n := 0;
  for i:=0 to FData.Count-1 do if not( trim(FData.Strings[0])='' ) then inc(n);

  SetLength(Line, n);

  for i:=0 to n-1 do
    if not( trim(FData.Strings[0])='' ) then begin
      new(Line[i]);
      s := FData.Strings[i];

      k := pos(',', s);
      val(copy(s, 1, k-1), v, j);
      Line[i].Left := v;

      delete(s, 1, k); k := pos(',', s);
      val(copy(s, 1, k-1), v, j);
      Line[i].Top := v;

      delete(s, 1, k); k := pos(',', s);
      val(copy(s, 1, k-1), v, j);
      Line[i].Right := v;

      delete(s, 1, k);
      val(s, v, j);
      Line[i].Bottom := v;
    end;
    
  RePaint;
end;

procedure TPolyLine.SaveSetupData;
var
  i: integer;
begin
  FData.Clear;
  for i:=0 to length(Line)-1 do
    FData.Add( format('%d,%d,%d,%d', [
        Line[i].Left,
        Line[i].Top,
        Line[i].Right,
        Line[i].Bottom
        ]));
end;


procedure TPolyLine.Loaded;
begin
  inherited;
  LoadSetupData;
end;



procedure TPolyLine.FreeLines;
var
  i: integer;
begin
  for i:=0 to length(Line)-1 do
    Dispose(Line[i]);
end;



{ TPolyLineEditor }

procedure TPolyLineEditor.Edit;
begin
  inherited;
  PolyLineSetupForm := TPolyLineSetupForm.Create(nil);

  PolyLineSetupForm.M := Component as TPolyLine;
  PolyLineSetupForm.D := Self;
  if PolyLineSetupForm.ShowModal = mrOk then
    PolyLineSetupForm.M.SaveSetupData
  else
    PolyLineSetupForm.M.LoadSetupData;

  PolyLineSetupForm.M.FlashLine := -1;
  PolyLineSetupForm.M.Repaint;
    
  PolyLineSetupForm.Free;
end;


end.
