unit SimpleImage;

interface

uses
//{$ifdef VER150}
  DesignIntf, DesignEditors, DesignWindows, DsnConst,
//{$else}
//  DsgnIntf,
//{$endif}
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, stdctrls, RpVisualGlobal, ScadaBase;


type
  TSimpleImage = class(TWinControl)
  private
    FData: TStringList;
    FFramed: boolean;

    procedure FreeLabels;
    procedure SetupData(Value: TStringList);
    procedure LoadSetupData;
    procedure SaveSetupData;
    procedure SetFramed(Value: boolean);
  protected
    procedure OnSimpleImageClick(Sender: TObject);
    procedure Loaded; override;
  public
    TagName: TTagName;
    ImagePic: TPicName;

    ImageObj: TImage;
    ShapeObj: TShape;

    Count: integer;
    Labels: array of TLabel;
    Checked: boolean;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

  published
    property Framed: Boolean read FFramed write SetFramed;
    property Data: TStringList read FData write SetupData;
    property OnClick;
  end;


  TSimpleImageEditor = class(TComponentEditor)
  private
  public
    procedure Edit; override;
  end;



procedure Register;

implementation
uses hyperstr, numbers, ExtDlgs, SimpleImageSetup;

procedure Register;
begin
  RegisterComponents('Scada', [TSimpleImage]);
  RegisterComponentEditor(TSimpleImage, TSimpleImageEditor);
end;

{ TSimpleImage }

procedure TSimpleImage.OnSimpleImageClick(Sender: TObject);
begin
  if Assigned(OnClick) then OnClick(Self);
end;

constructor TSimpleImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 100;
  Height := 100;
  ShowHint := true;

  FData := TStringList.Create;

  Labels := nil;
  Count := 0;

  //  
  ImageObj := TImage.Create(Self);
  ImageObj.Parent := self;
  ImageObj.Align := alClient;
  ImageObj.Stretch := true;
  ImageObj.Visible := true;

  ShapeObj := TShape.Create(Self);
  ShapeObj.Parent := self;
  ShapeObj.Align := alClient;
  ShapeObj.Visible := true;
  ShapeObj.Brush.Style := bsClear;
  ShapeObj.Pen.Width := 2;

  ImageObj.OnClick := OnSimpleImageClick;
end;

destructor TSimpleImage.Destroy;
begin
  FreeLabels;
  inherited;
end;

procedure TSimpleImage.SetupData(Value: TStringList);
begin
  FData.Assign(Value);
  LoadSetupData;
end;




procedure TSimpleImage.LoadSetupData;
var
  s: string;
  k, i, j, v: integer;
begin

  // TagPath
  if FData.Count>0 then TagName := FData.Strings[0] else TagName := '';

  // Image
  if FData.Count>1 then ImagePic := FData.Strings[1] else ImagePic := '';
  if FileExists(VisImagesPath + ImagePic) then
    ImageObj.Picture.LoadFromFile(VisImagesPath + ImagePic);

  // Labels
  FreeLabels;
  if FData.Count>2 then s := FData.Strings[2] else s := '';
  val(s, v, j);
  Count := v;
  SetLength(Labels, Count);

  for i:=0 to Count-1 do begin
    Labels[i] := TLabel.Create(self);
    Labels[i].Parent := self;
    Labels[i].Visible := true;
    Labels[i].Transparent := true;
    Labels[i].AutoSize := false;
    Labels[i].OnClick := OnSimpleImageClick;

    if FData.Count>3+i then s := FData.Strings[3+i] else s := '';

    k := 1;
    val(Parse(s, ',', k), v, j);
    Labels[i].Left := v;

    val(Parse(s, ',', k), v, j);
    Labels[i].Top := v;

    val(Parse(s, ',', k), v, j);
    Labels[i].Width := v;

    val(Parse(s, ',', k), v, j);
    Labels[i].Height := v;

    val(Parse(s, ',', k), v, j);
    Labels[i].Alignment := TAlignment(v);

    val(Parse(s, ',', k), v, j);
    Labels[i].Font.Size := v;

    val(Parse(s, ',', k), v, j);
    Labels[i].Font.Color := TColor(v);

    val(Parse(s, ',', k), v, j);
    if (v and 1) > 0 then Labels[i].Font.Style := Labels[i].Font.Style + [fsBold];
    if (v and 2) > 0 then Labels[i].Font.Style := Labels[i].Font.Style + [fsItalic];

    Labels[i].Font.Name := Parse(s, ',', k);
    
    Labels[i].Caption := Parse(s, ',', k);
  end;

end;

procedure TSimpleImage.SaveSetupData;
var
  i,font_style: integer;
begin
  FData.Clear;
  FData.Add(TagName);
  FData.Add(ImagePic);

  FData.Add(IntToStr(Count));
  for i:=0 to Count-1 do begin
    font_style := iif( fsBold in Labels[i].Font.Style, 1, 0) +
          iif( fsItalic in Labels[i].Font.Style, 2, 0);

    FData.Add( format('%d,%d,%d,%d,%d,%d,%d,%d,%s,%s',
        [
          Labels[i].Left, Labels[i].Top,
          Labels[i].Width, Labels[i].Height,
          integer(Labels[i].Alignment),
          Labels[i].Font.Size,
          integer(Labels[i].Font.Color),
          font_style,
          Labels[i].Font.Name,
          Labels[i].Caption
        ]));
  end;

end;


procedure TSimpleImage.Loaded;
begin
  inherited;
  LoadSetupData;
end;


procedure TSimpleImage.FreeLabels;
var
  i: integer;
begin
  for i:=0 to Count-1 do begin
    Labels[i].Free;
  end;
  Labels := nil;
end;



procedure TSimpleImage.SetFramed(Value: boolean);
begin
  FFramed := value;
  ShapeObj.Visible := value;
end;

{ TSimpleImageEditor }


procedure TSimpleImageEditor.Edit;
begin
  inherited;
  SimpleImageSetupForm := TSimpleImageSetupForm.Create(nil);

  SimpleImageSetupForm.M := Component as TSimpleImage;
  SimpleImageSetupForm.D := Self;
  if SimpleImageSetupForm.ShowModal = mrOk then
    SimpleImageSetupForm.M.SaveSetupData
  else
    SimpleImageSetupForm.M.LoadSetupData;

  SimpleImageSetupForm.Free;
end;





end.
