unit tvGroup;

interface
uses
  Graphics, pFIBDataSet, pFIBDatabase, DbChart, Series, sysutils, FIBDatabase, pFIBProps;

type
  TtvTag = class
    Name: string;
    Descr: string;
    LegName: string;
    Transform: string;

    Width: integer;
    UseColor: boolean;
    Color: TColor;
    Stairs: boolean;
    Pointers: boolean;

    function GetLegname: string;
  end;

  TtvTrendTag = class(TtvTag)
  public
    IdTag: integer;
    ArcName: string;

    ds: TpFIBDataset;
    trn: TpFIBTransaction;
    srs: TLineSeries;

    constructor Create(db: TpFIBDatabase; chart: TDBChart);
    destructor Destroy; override;
    procedure Init(OptimizedSelect: boolean);
    procedure Open(dtBeg, dtEnd: TDatetime);
  end;

  TtvGroup = class
    Name: string;
    UnitName: string;
    Descr: string;
    Tags: array of TtvTag;
    UseColor: boolean;
    Color: TColor;
    DefaultDelta: TDatetime;
    DefaultYmin: integer;
    DefaultYmax: integer;

    dtBeg, dtEnd: TDatetime;
    Ymax, Ymin: double;

    procedure Clear;
    destructor Destroy; override;
  end;



implementation

uses DB;

{ TtvGroup }

procedure TtvGroup.Clear;
var
  i: integer;
begin
  for i:=0 to length(Tags)-1 do Tags[i].Free;
  Tags := nil;
end;


destructor TtvGroup.Destroy;
begin
  Clear;
  inherited;
end;

{ TtvTrendTag }

constructor TtvTrendTag.Create(db: TpFIBDatabase; chart: TDBChart);
begin
  trn := TpFIBTransaction.Create(nil);
  trn.TRParams.Add('read');
  trn.TRParams.Add('read_committed');
  trn.TRParams.Add('rec_version');
  trn.TRParams.Add('nowait');
  trn.TimeoutAction := TARollback;
  trn.TPBMode := tpbDefault;
  
  trn.DefaultDatabase := db;
  ds := TpFIBDataSet.Create(nil);
  ds.Database := db;
  ds.Transaction := trn;

  srs := TLineSeries.Create(nil);
  srs.Title := '';
  srs.ParentChart := Chart;
  srs.Marks.Visible := false;
  srs.Pointer.Style := psCircle;


end;

destructor TtvTrendTag.Destroy;
begin

  try
    if ds.Active then
      ds.Close;
    if trn.Active then
      trn.Commit;
  except
  end;

  srs.Free;
  ds.Free;
  trn.Free;
  inherited;
end;

procedure TtvTrendTag.Init(OptimizedSelect: boolean);
var
  s,ss,trns: string;
  i: integer;
begin
  if Width=0 then
    srs.LinePen.Width := 1
  else
    srs.LinePen.Width := Width;
  srs.Stairs := Stairs;
  srs.Pointer.Visible := Pointers;
  if UseColor then
    srs.SeriesColor := Color;

  srs.Pointer.VertSize := srs.LinePen.Width;
  srs.Pointer.HorizSize := srs.LinePen.Width;
  srs.Pointer.Pen.Color := srs.SeriesColor;

  srs.Title := GetLegname;


  trn.Active := false;
  ds.SelectSQL.Text :=
      'select t.idtag, a.arcname from taglist t, arclist a where t.idarc=a.idarc '+
      'and t.tagname=''' + Name + '''';

  ds.Open;
  if not ds.Eof then begin
    idtag := ds.Fields[0].AsInteger;
    arcname := ds.Fields[1].AsString;
  end else
    idtag := -1;
  ds.Close;

  if idtag <> -1 then begin
  
    if OptimizedSelect then
      s := 'FY'
    else
      s := 'T'+inttostr(IdTag);

    if not( trim(Transform)='' ) then begin
      i := 1;
      trns := Transform;
      while i<length(trns) do begin
        if trns[i]='%' then begin
          case trns[i+1] of
            'i','I': ss:=s;
            'd','D': ss:='CAST(' + s + ' AS FLOAT)';
            '%': s:='%';
          else
            ss:=''
          end;

          delete(trns, i, 2);
          insert(ss, trns, i);
        end;
        inc(i);
      end;
      s := trns;
    end;

    if OptimizedSelect then
      ds.SelectSQL.Text :=
          'select DT, '+s+' as FY from gettrendexa(:DTBEG, :DTEND, '''+Name+''','''+ArcName+''',1000)'
    else
      ds.SelectSQL.Text :=
          'select DT, '+s+' as FY from '+ArcName+' where (DT>:DTBEG) AND (DT<:DTEND)';

    srs.DataSource := ds;
    srs.YValues.ValueSource := 'FY';
    srs.XValues.ValueSource := 'DT';
    srs.XValues.DateTime := true;
    srs.CheckDatasource;

    try
      ds.Transaction.StartTransaction;
      ds.Prepare;
    except
    end;
  end;

end;

procedure TtvTrendTag.Open(dtBeg, dtEnd: TDatetime);
begin
  if idtag = -1 then exit;
  if trn.Active then trn.Commit;

  ds.ParamByName('DTBEG').AsDateTime := dtBeg;
  ds.ParamByName('DTEND').AsDateTime := dtEnd;

  try
    ds.Transaction.StartTransaction;
    ds.Open;
  except
  end;

end;

{ TtvTag }

function TtvTag.GetLegname: string;
begin
  if not(trim(LegName)='') then
    result := LegName
  else
  if not(trim(Descr)='') then
    result := Descr
  else
    result := Name;
end;

end.
