unit VisNumericPanel;

interface

uses
  DesignIntf, DesignEditors,
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, stdctrls, uPSComponent, uPSCompiler, uPSRuntime,
  VisControl;

type
  TVisNumericPanel = class(TVisControl)
    procedure PSScriptCompile(Sender: TPSScript);
    procedure PSScriptCompImport(Sender: TObject; x: TPSPascalCompiler);
    procedure PSScriptExecImport(Sender: TObject; se: TPSExec; x: TPSRuntimeClassImporter);
    procedure PSScriptExecute(Sender: TPSScript);
  private
    FCustomClick: TNotifyEvent;
    FPanel: TPanel;
    FAutoUpdateApply: boolean;
    FApplying: boolean;
    FValueMul: integer;
    FValueDiv: integer;
    FPrecision: integer;
    FReadOnly: boolean;

    FUseSaveMessage: boolean;
    FSaveMessageStr: string;

//    FHint: string;
  protected
    procedure OnControlClick(Sender: TObject);
    procedure ScriptProcess(ForceExecute: boolean);
    procedure onTagStatusChanged(good: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Init; override;
    procedure Process; override;
    function IsApplying: boolean;

    procedure setcolors(AFontColor: integer; ABackColor: integer);

    function CompileScript(var Messages: string): boolean; override;
    procedure ExecuteScript; override;

    procedure DoApply;
    procedure DoUpdate;

  published
    property UseSaveMessage: boolean read FUseSaveMessage write FUseSaveMessage;
    property SaveMessageStr: string read FSaveMessageStr write FSaveMessageStr;
    property AutoUpdateApply: boolean read FAutoUpdateApply write FAutoUpdateApply;

    property ReadOnly: boolean read FReadOnly write FReadOnly;
    property ValueMul: integer read FValueMul write FValueMul;
    property ValueDiv: integer read FValueDiv write FValueDiv;
    property Precision: integer read FPrecision write FPrecision;

    property objPanel: TPanel read FPanel;

    property OnCustomClick: TNotifyEvent read FCustomClick write FCustomClick;
    property Align;
    property Anchors;
    property Enabled;
    property Visible;
  end;

//  TVisNumericPanel_ScriptPropEdit =class(TClassProperty)
//  public
//    function GetAttributes: TPropertyAttributes; override;
//    procedure Edit; override;
//  end;

procedure Register;

implementation
uses
  uPSR_std,
  uPSC_std,
  uPSR_stdctrls,
  uPSC_stdctrls,
  uPSR_extctrls,
  uPSC_extctrls,
  uPSR_forms,
  uPSC_forms,
  uPSC_graphics,
  uPSC_controls,
  uPSC_classes,
  uPSR_graphics,
  uPSR_controls,
  uPSR_classes,
  uPSR_visscada,
  uPSC_visscada,
  TagStorage,
  numbers,
  VisControlScriptEdit,
  MesConst,
  meslogging,
  InputDialog,
  RpVisualMain,
  RpVisualGlobal,
  RpVisualUtils,
  hyperstr;

var
  PSScript: TPSScript;
  PSScript_refcount: integer;
  PSVisNumericPanel: TVisNumericPanel;

  tmp_compile_src, tmp_compile_out: string;

procedure Register;
begin
  RegisterComponents('Scada', [TVisNumericPanel]);
  RegisterComponentEditor(TVisNumericPanel, TVisControlEditor);
end;

{ TVisualDevice }

constructor TVisNumericPanel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ShowHint := true;

  //  
  FPanel := TPanel.Create(Self);
  FPanel.Name := 'objPanel';
  FPanel.Caption := '';
  FPanel.Parent := self;
  FPanel.Visible := true;
  FPanel.SetSubComponent(true);
  FPanel.OnClick := OnControlClick;
  FPanel.Anchors := [akLeft, akTop, akRight, akBottom];
  FPanel.BevelOuter := bvLowered;
  FPanel.DoubleBuffered := true;

  Width := FPanel.Width;
  Height := FPanel.Height;

  FAutoUpdateApply := true;
  FValueMul := 1;
  FValueDiv := 1;
  FPrecision := 0;
  FReadOnly := false;

//  FHint := '';

  if PSScript = nil then begin
    PSScript := TPSScript.Create(nil);
    PSScript.OnCompile := PSScriptCompile;
    PSScript.OnCompImport := PSScriptCompImport;
    PSScript.OnExecImport := PSScriptExecImport;
    PSScript.OnExecute := PSScriptExecute;
    inc(PSScript_refcount);
  end;

  FPanel.OnMouseMove := OnMouseMove;
end;


// INIT
procedure TVisNumericPanel.Init;
var
  s, _tags: string;
  i,k: integer;
  use_path: boolean;
begin
  inherited;

  //  Cursor := crHandPoint;
  objPanel.Cursor := Cursor;

  LostTags := '';

  _tags := '';
  TagRoot := trim(TagRoot);
  TagIdRoot := trim(TagIdRoot);
  TagPath := trim(TagPath);

  TagPath := ReplaceStr(TagPath, TagIdRoot, TagRoot);
  
  SetLength(Tags, FTagNames.Count);

  IsConnectionBad := false;
  for i:=0 to FTagNames.Count-1 do begin

    s:=FTagNames[i];

    use_path := true;
    if copy(s,1,1)='$' then begin
      use_path := false;
      delete(s,1,1);
    end;

    if length(TagRoot)>0 then begin
      k:=pos(TagIdRoot, s);
      if k>0 then begin
        delete(s, k, length(TagIdRoot));
        insert(TagRoot, s, k);
      end;
    end;

    if (use_path) and (length(TagPath)>0) then begin
      s := TagPath + '.' + s;
    end;

    _tags := _tags + s + ' ';
    Tags[i].idx := GetTagIndexWithLost(s, LostTags);
    Tags[i].value := 0;
    Tags[i].good := true;

    if Tags[i].idx < 0 then
      _tags := _tags + s + '- NOT FOUND! ';

    IsConnectionBad := (IsConnectionBad) or (Tags[i].idx < 0);
  end;

  CompileScript(s);
  for i:=0 to length(Tags)-1 do begin
    Tags[i].value := -1;
    Tags[i].empty := true;
  end;

  FSaveMessageStr := ReplaceStr(FSaveMessageStr, TagIdRoot, TagRoot);

  ScriptProcess(true);
end;


// PROCESS HANDLER
procedure TVisNumericPanel.Process;
begin
  if FAutoUpdateApply then begin
    FApplying := false;
    ScriptProcess(false);
    ExecuteProcessEvent;
  end;
end;

procedure TVisNumericPanel.DoApply;
begin
  FApplying := true;
  ScriptProcess(true);
end;

procedure TVisNumericPanel.DoUpdate;
begin
  FApplying := false;
  ScriptProcess(true);
end;

function TVisNumericPanel.IsApplying: boolean;
begin
  result := FApplying;
end;

procedure TVisNumericPanel.ScriptProcess(ForceExecute: boolean);
var
  i: integer;
  fl: boolean;
  r: single;
begin
//  try
  IsValueBad := false;
  fl := updateTags();

  if (fl) or (not FExecuteOnTagChange) or (FApplying) or (ForceExecute) then begin

    if FScriptDefEnabled then
    begin
      //    ( )
      if FApplying then begin

        val(objPanel.Caption, r, i);
        r := r * ValueDiv;
        r := r / ValueMul;
        settagvaldouble(0, r);
      end else begin
        if isTagNan(0) then
          objPanel.Caption := '---'
        else
        if ValueDiv<>0 then
          objPanel.Caption := FloatStr(gettagvaldouble(0) * ValueMul  / ValueDiv, 15, Precision)
        else
          objPanel.Caption := 'DivByZero';
      end;
    end;

    if FScriptEnabled then
      ExecuteScript;
  end;
//  except
//    ShowMessage(name + ' ' + TagNames.Text);
//  end;
end;


procedure TVisNumericPanel.onTagStatusChanged(good: Boolean);
begin
  if (not FBadColorsEnabled) then  exit;

  if good then begin
    objPanel.Color := FGoodColorBack1;
    objPanel.Font.Color := FGoodColorFont;
  end else begin
    putColorGood(objPanel.Font.Color, objPanel.Color, 0);
    objPanel.Color := FBadColorBack;
    objPanel.Font.Color := FBadColorFont;
  end;
end;


procedure TVisNumericPanel.OnControlClick(Sender: TObject);
var
  s: string;
begin
  if not rvgLogIn then
    exit;

  _ShowLostTags(self);

  if Assigned(FCustomClick) then begin
    FCustomClick(self);
  end;

  if FReadonly then exit;

  s := objPanel.Caption;

  if not ShowInputDialog( get_clear_hint(Hint), ' ', s) then
//  if not InputQuery( get_clear_hint(Hint), ' ', s) then
    exit;
  objPanel.Caption := s;

  if UseSaveMessage then
    SaveMessageText(mcParam_text, SaveMessageStr, get_clear_hint(Hint) + ' = ' + s, mcParam_bc, mcParam_fc);

  if FAutoUpdateApply then DoApply;
  OnDeviceClick(sender);
end;


procedure TVisNumericPanel.setcolors(AFontColor: integer; ABackColor: integer);
begin
  objPanel.Font.Color := AFontColor;
  objPanel.Color := ABackColor;
end;


//==============================================================================
procedure TVisNumericPanel.PSScriptCompImport(Sender: TObject; x: TPSPascalCompiler);
begin
  SIRegister_Std(x);
  SIRegister_Classes(x, true);
  SIRegister_Graphics(x, true);
  SIRegister_Controls(x);
  SIRegister_stdctrls(x);
  SIRegister_extctrls(x);
  SIRegister_Forms(x);
  SIRegister_visscada(x);
end;

procedure TVisNumericPanel.PSScriptExecImport(Sender: TObject; se: TPSExec;
  x: TPSRuntimeClassImporter);
begin
  RIRegister_Std(x);
  RIRegister_Classes(x, True);
  RIRegister_Graphics(x, True);
  RIRegister_Controls(x);
  RIRegister_stdctrls(x);
  RIRegister_extctrls(x);
  RIRegister_Forms(x);
  RIRegister_visscada(x);
end;

procedure TVisNumericPanel.PSScriptCompile(Sender: TPSScript);
begin
  Sender.AddRegisteredPTRVariable('self', 'TVisNumericPanel');
end;

procedure TVisNumericPanel.PSScriptExecute(Sender: TPSScript);
begin
  Sender.SetPointerToData('self', @PSVisNumericPanel, sender.FindNamedType('TVisNumericPanel'));
end;

function TVisNumericPanel.CompileScript(var Messages: string): boolean;
var
  i: integer;
begin
  if SameText(tmp_compile_src, Script.Text) then begin
    CompiledScript := tmp_compile_out;
    IsCompileScriptOk := true;
  end else begin
    PSVisNumericPanel := self;
    PSScript.Script.Text := Script.Text;
    IsCompileScriptOk := PSScript.Compile;

    if IsCompileScriptOk then begin
      PSScript.GetCompiled(CompiledScript);
      tmp_compile_src := Script.Text;
      tmp_compile_out := CompiledScript;
    end;

    Messages := '';
    for i:=0 to PSScript.CompilerMessageCount-1 do
      Messages := Messages + PSScript.CompilerMessages[i].MessageToString + #13#10;
  end;

  if IsCompileScriptOk then
    Messages := Messages + 'Successfully compiled!'#13#10;

  result := True;
end;

procedure TVisNumericPanel.ExecuteScript;
begin
  if IsCompileScriptOk then begin
    PSVisNumericPanel := self;
    PSScript.SetCompiled(CompiledScript);
    PSScript.Execute;
  end;
end;




initialization
  PSScript := nil;
  PSScript_refcount := 0;

  tmp_compile_src := '*';
  tmp_compile_out := '';

finalization
  dec(PSScript_refcount);
  if PSScript_refcount = 0 then begin
    PSScript.Free;
    PSScript := nil;
  end;

end.
